package App::Ground;

our $VERSION =  '0.01_03';


1;

=encoding utf8

=head1 NAME

C<App::Ground> - Ground your application.

=head1 DESCRIPTION

In circuit design, "ground" (GND) is a shared reference node (often called
"0V") and a current return path. It is convenient because you can connect to it
from many places, but it is not magic: real circuits may have multiple grounds,
local grounds, and bad grounding can create noise or loops.

C<App::Ground> uses the same metaphor for programming: it provides a small set
of modules that act like a "ground" for infrastructure dependencies (logging,
configuration, application handle, optional request context). This reduces
parameter plumbing, but it should be used for infrastructure, not business
data.

This is the classic trade-off of I<ambient authority> vs I<dependency
injection>: globals are convenient, DI is explicit.

=head1 EXAMPLES IN OTHER LANGUAGES

=head2 JavaScript / Node.js

  console.log('hello');           // global logging sink
  const port = process.env.PORT;  // global-ish configuration

=head2 Rust

  log::info!("hello");  // global logger backend set once at startup

=head2 Go

  // context.Context: one value passed through the stack, used by deep code.

=head2 Node.js (AsyncLocalStorage)

  // per-async-chain "ambient" context (request id, trace ids, and so on).

=head2 Java

  private static final Logger LOG = LoggerFactory.getLogger(X.class);
  // Per-request context is often stored in ThreadLocal / MDC.

=head1 USAGE (THIS DISTRO)

This distribution ships several tiny "ground" modules.

=head2 C - Context

  use C;
  C::app($app);

  my $db  =  C::db;
  my $cfg =  C::config;

Optional request context (for example a Mojolicious controller) can be stored
in the package variable C<$C::C> and read via C<C::C()>:

  sub handler {
    my ($c) = @_;
    local $C::C = $c;
    deep_call();
    $c->render(text => 'ok');
  }

  sub deep_call {
    my $c = C::C() or return;
    # ...
  }

B<Note>: dynamic scoping (C<local>) does not automatically cross async or
promise boundaries. If you defer work, capture the context and re-localize it
in the callback.

=head2 L - Logging

  use L;
  L::info "Hello\\n";

To replace the logger implementation (for example with Mojo::Log):

  $L::logger = sub { $app->log };

You can also override it temporarily:

  local $L::logger = sub { My::Test::Logger->new };

=head2 U - Utilities

C<U> utilities use C<C> for configuration and C<L> for logging. For example,
C<U::ewarn> reads C<C::config-E<gt>{Debug}{to}> and sends emails via
L<Email::Stuffer>.

=head1 AUTHOR

Eugen Konkov

=head1 SEE ALSO

L<A>, L<C>, L<I>, L<L>, L<M>, L<S>, L<T>, L<U>.

=cut
