#define PERL_NO_GET_CONTEXT
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "xs_jit_builder.h"

/* Store the builder pointer in the object's hash */
#define BUILDER_KEY "_builder_ptr"
#define BUILDER_KEY_LEN 12

static XS_JIT_Builder* get_builder(pTHX_ SV* self) {
    HV* hv;
    SV** svp;
    
    if (!SvROK(self) || SvTYPE(SvRV(self)) != SVt_PVHV) {
        croak("Not a valid XS::JIT::Builder object");
    }
    
    hv = (HV*)SvRV(self);
    svp = hv_fetch(hv, BUILDER_KEY, BUILDER_KEY_LEN, 0);
    
    if (!svp || !SvOK(*svp)) {
        croak("Builder object not initialized");
    }
    
    return INT2PTR(XS_JIT_Builder*, SvIV(*svp));
}

static void set_builder(pTHX_ SV* self, XS_JIT_Builder* b) {
    HV* hv = (HV*)SvRV(self);
    hv_store(hv, BUILDER_KEY, BUILDER_KEY_LEN, newSViv(PTR2IV(b)), 0);
}

MODULE = XS::JIT::Builder  PACKAGE = XS::JIT::Builder

PROTOTYPES: DISABLE

BOOT:
{
    HV* stash = gv_stashpv("XS::JIT::Builder", GV_ADD);
    
    /* Inline op type constants */
    newCONSTSUB(stash, "INLINE_NONE",      newSViv(XS_JIT_INLINE_NONE));
    newCONSTSUB(stash, "INLINE_GETTER",    newSViv(XS_JIT_INLINE_GETTER));
    newCONSTSUB(stash, "INLINE_SETTER",    newSViv(XS_JIT_INLINE_SETTER));
    newCONSTSUB(stash, "INLINE_HV_GETTER", newSViv(XS_JIT_INLINE_HV_GETTER));
    newCONSTSUB(stash, "INLINE_HV_SETTER", newSViv(XS_JIT_INLINE_HV_SETTER));
}

SV*
new(class, ...)
    const char* class
    PREINIT:
        HV* hv;
        SV* self;
        XS_JIT_Builder* b;
        int indent_width = 4;
        int use_tabs = 0;
        int i;
    CODE:
        /* Parse options */
        for (i = 1; i < items; i += 2) {
            if (i + 1 < items) {
                const char* key = SvPV_nolen(ST(i));
                if (strEQ(key, "indent_width")) {
                    indent_width = SvIV(ST(i + 1));
                } else if (strEQ(key, "use_tabs")) {
                    use_tabs = SvTRUE(ST(i + 1)) ? 1 : 0;
                }
            }
        }
        
        /* Create the builder */
        b = xs_jit_builder_new(aTHX);
        xs_jit_set_indent_width(b, indent_width);
        xs_jit_set_use_tabs(b, use_tabs);
        
        /* Create blessed hashref */
        hv = newHV();
        self = newRV_noinc((SV*)hv);
        sv_bless(self, gv_stashpv(class, GV_ADD));
        
        /* Store the pointer */
        set_builder(aTHX_ self, b);
        
        RETVAL = self;
    OUTPUT:
        RETVAL

void
DESTROY(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_builder_free(aTHX_ b);

SV*
code(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        RETVAL = xs_jit_builder_code(aTHX_ b);
    OUTPUT:
        RETVAL

SV*
reset(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_builder_reset(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
line(self, fmt, ...)
    SV* self
    const char* fmt
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_line(aTHX_ b, "%s", fmt);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
raw(self, text)
    SV* self
    const char* text
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_raw(aTHX_ b, "%s", text);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
comment(self, text)
    SV* self
    const char* text
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_comment(aTHX_ b, text);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
blank(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_blank(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
indent(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_indent(b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
dedent(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_dedent(b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
block_start(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_block_start(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
block_end(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_block_end(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
xs_function(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xs_function(aTHX_ b, name);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
xs_preamble(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xs_preamble(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
xs_end(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xs_end(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
xs_return(self, count)
    SV* self
    int count
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xs_return(aTHX_ b, count);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
xs_return_undef(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xs_return_undef(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_undef(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xs_return_undef(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
croak_usage(self, usage)
    SV* self
    const char* usage
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_croak_usage(aTHX_ b, usage);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
if(self, cond)
    SV* self
    const char* cond
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_if(aTHX_ b, cond);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
elsif(self, cond)
    SV* self
    const char* cond
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_elsif(aTHX_ b, cond);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
else(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_else(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
endif(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_endif(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
for(self, init, cond, incr)
    SV* self
    const char* init
    const char* cond
    const char* incr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_for(aTHX_ b, init, cond, incr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
while(self, cond)
    SV* self
    const char* cond
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_while(aTHX_ b, cond);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
endloop(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_endloop(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
endfor(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_endloop(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
endwhile(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_endloop(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
block(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_block_start(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
endblock(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_block_end(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare(self, type, name, ...)
    SV* self
    const char* type
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
        const char* value = NULL;
    CODE:
        if (items > 3 && SvOK(ST(3))) {
            value = SvPV_nolen(ST(3));
        }
        b = get_builder(aTHX_ self);
        xs_jit_declare(aTHX_ b, type, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_sv(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_sv(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_hv(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_hv(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_av(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_av(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
new_hv(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_hv(aTHX_ b, name, "newHV()");
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
new_av(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_av(aTHX_ b, name, "newAV()");
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_int(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_int(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_iv(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_iv(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_nv(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_nv(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
declare_pv(self, name, value)
    SV* self
    const char* name
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_declare_pv(aTHX_ b, name, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
assign(self, var, value)
    SV* self
    const char* var
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_assign(aTHX_ b, var, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
get_self(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_get_self(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
get_self_hv(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_get_self_hv(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
get_self_av(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_get_self_av(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
hv_fetch(self, hv, key, len, result_var)
    SV* self
    const char* hv
    const char* key
    STRLEN len
    const char* result_var
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_hv_fetch(aTHX_ b, hv, key, len, result_var);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
hv_fetch_sv(self, hv, key_expr, len_expr, result_var)
    SV* self
    const char* hv
    const char* key_expr
    const char* len_expr
    const char* result_var
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_hv_fetch_sv(aTHX_ b, hv, key_expr, len_expr, result_var);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
hv_store(self, hv, key, len, value)
    SV* self
    const char* hv
    const char* key
    STRLEN len
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_hv_store(aTHX_ b, hv, key, len, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
hv_store_sv(self, hv, key_expr, len_expr, value)
    SV* self
    const char* hv
    const char* key_expr
    const char* len_expr
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_hv_store_sv(aTHX_ b, hv, key_expr, len_expr, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
hv_fetch_return(self, hv, key, len)
    SV* self
    const char* hv
    const char* key
    STRLEN len
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_hv_fetch_return(aTHX_ b, hv, key, len);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
av_fetch(self, av, index, result_var)
    SV* self
    const char* av
    const char* index
    const char* result_var
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_av_fetch(aTHX_ b, av, index, result_var);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
av_store(self, av, index, value)
    SV* self
    const char* av
    const char* index
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_av_store(aTHX_ b, av, index, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
av_push(self, av, value)
    SV* self
    const char* av
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_av_push(aTHX_ b, av, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
av_len(self, av, result_var)
    SV* self
    const char* av
    const char* result_var
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_av_len(aTHX_ b, av, result_var);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
new_sv_iv(self, result_var, value)
    SV* self
    const char* result_var
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_new_sv_iv(aTHX_ b, result_var, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
new_sv_nv(self, result_var, value)
    SV* self
    const char* result_var
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_new_sv_nv(aTHX_ b, result_var, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
new_sv_pv(self, result_var, str, len)
    SV* self
    const char* result_var
    const char* str
    STRLEN len
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_new_sv_pv(aTHX_ b, result_var, str, len);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
mortal(self, sv)
    SV* self
    const char* sv
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_mortal(aTHX_ b, sv);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
croak(self, message)
    SV* self
    const char* message
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_croak(aTHX_ b, message);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
warn(self, message)
    SV* self
    const char* message
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_warn(aTHX_ b, message);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
check_items(self, min, max, usage)
    SV* self
    int min
    int max
    const char* usage
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_check_items(aTHX_ b, min, max, usage);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
check_defined(self, sv, error_msg)
    SV* self
    const char* sv
    const char* error_msg
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_check_defined(aTHX_ b, sv, error_msg);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
check_hashref(self, sv, error_msg)
    SV* self
    const char* sv
    const char* error_msg
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_check_hashref(aTHX_ b, sv, error_msg);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
check_arrayref(self, sv, error_msg)
    SV* self
    const char* sv
    const char* error_msg
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_check_arrayref(aTHX_ b, sv, error_msg);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
sv_to_iv(self, result_var, sv)
    SV* self
    const char* result_var
    const char* sv
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_sv_to_iv(aTHX_ b, result_var, sv);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
sv_to_nv(self, result_var, sv)
    SV* self
    const char* result_var
    const char* sv
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_sv_to_nv(aTHX_ b, result_var, sv);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
sv_to_pv(self, result_var, len_var, sv)
    SV* self
    const char* result_var
    SV* len_var
    const char* sv
    PREINIT:
        XS_JIT_Builder* b;
        const char* len_var_str = NULL;
    CODE:
        if (SvOK(len_var)) {
            len_var_str = SvPV_nolen(len_var);
        }
        b = get_builder(aTHX_ self);
        xs_jit_sv_to_pv(aTHX_ b, result_var, len_var_str, sv);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
sv_to_bool(self, result_var, sv)
    SV* self
    const char* result_var
    const char* sv
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_sv_to_bool(aTHX_ b, result_var, sv);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_iv(self, value)
    SV* self
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_return_iv(aTHX_ b, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_nv(self, value)
    SV* self
    const char* value
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_return_nv(aTHX_ b, value);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_pv(self, str, ...)
    SV* self
    const char* str
    PREINIT:
        XS_JIT_Builder* b;
        const char* len_str = NULL;
    CODE:
        if (items > 2 && SvOK(ST(2))) {
            len_str = SvPV_nolen(ST(2));
        }
        b = get_builder(aTHX_ self);
        xs_jit_return_pv(aTHX_ b, str, len_str);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_sv(self, sv)
    SV* self
    const char* sv
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_return_sv(aTHX_ b, sv);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_yes(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_return_yes(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_no(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_return_no(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
return_self(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_return_self(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
method_start(self, func_name, min_args, max_args, usage)
    SV* self
    const char* func_name
    int min_args
    int max_args
    const char* usage
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_method_start(aTHX_ b, func_name, min_args, max_args, usage);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
ro_accessor(self, func_name, attr_name, attr_len)
    SV* self
    const char* func_name
    const char* attr_name
    STRLEN attr_len
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ro_accessor(aTHX_ b, func_name, attr_name, attr_len);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
rw_accessor(self, func_name, attr_name, attr_len)
    SV* self
    const char* func_name
    const char* attr_name
    STRLEN attr_len
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_rw_accessor(aTHX_ b, func_name, attr_name, attr_len);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
accessor(self, attr_name, ...)
    SV* self
    const char* attr_name
    PREINIT:
        XS_JIT_Builder* b;
        int readonly = 0;
        STRLEN attr_len;
    CODE:
        b = get_builder(aTHX_ self);
        attr_len = strlen(attr_name);
        /* Check for readonly option in hash-style args */
        if (items > 2) {
            SV* opts = ST(2);
            if (SvROK(opts) && SvTYPE(SvRV(opts)) == SVt_PVHV) {
                HV* hv = (HV*)SvRV(opts);
                SV** val = hv_fetchs(hv, "readonly", 0);
                if (val && *val && SvTRUE(*val)) {
                    readonly = 1;
                }
            }
        }
        if (readonly) {
            xs_jit_ro_accessor(aTHX_ b, attr_name, attr_name, attr_len);
        } else {
            xs_jit_rw_accessor(aTHX_ b, attr_name, attr_name, attr_len);
        }
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
predicate(self, attr_name)
    SV* self
    const char* attr_name
    PREINIT:
        XS_JIT_Builder* b;
        char func_name[256];
        STRLEN attr_len;
    CODE:
        b = get_builder(aTHX_ self);
        attr_len = strlen(attr_name);
        snprintf(func_name, sizeof(func_name), "has_%s", attr_name);
        xs_jit_predicate(aTHX_ b, func_name, attr_name, attr_len);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
clearer(self, attr_name)
    SV* self
    const char* attr_name
    PREINIT:
        XS_JIT_Builder* b;
        char func_name[256];
        STRLEN attr_len;
    CODE:
        b = get_builder(aTHX_ self);
        attr_len = strlen(attr_name);
        snprintf(func_name, sizeof(func_name), "clear_%s", attr_name);
        xs_jit_clearer(aTHX_ b, func_name, attr_name, attr_len);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
constructor(self, func_name, attrs)
    SV* self
    const char* func_name
    AV* attrs
    PREINIT:
        XS_JIT_Builder* b;
        XS_JIT_Attr* attr_array;
        int num_attrs;
        int i;
    CODE:
        b = get_builder(aTHX_ self);
        num_attrs = av_len(attrs) + 1;
        
        if (num_attrs > 0) {
            Newx(attr_array, num_attrs, XS_JIT_Attr);
            
            for (i = 0; i < num_attrs; i++) {
                SV** elem = av_fetch(attrs, i, 0);
                if (elem && SvROK(*elem) && SvTYPE(SvRV(*elem)) == SVt_PVAV) {
                    AV* pair = (AV*)SvRV(*elem);
                    SV** name_sv = av_fetch(pair, 0, 0);
                    SV** len_sv = av_fetch(pair, 1, 0);
                    
                    if (name_sv && len_sv) {
                        attr_array[i].name = SvPV_nolen(*name_sv);
                        attr_array[i].len = SvIV(*len_sv);
                    }
                } else if (elem && SvPOK(*elem)) {
                    /* Simple string attribute name */
                    attr_array[i].name = SvPV_nolen(*elem);
                    attr_array[i].len = SvCUR(*elem);
                }
            }
            
            xs_jit_constructor(aTHX_ b, func_name, NULL, attr_array, num_attrs);
            Safefree(attr_array);
        } else {
            xs_jit_constructor(aTHX_ b, func_name, NULL, NULL, 0);
        }
        
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
op_ro_accessor(self, func_name, slot)
    SV* self
    const char* func_name
    IV slot
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_op_ro_accessor(aTHX_ b, func_name, slot);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
op_rw_accessor(self, func_name, slot)
    SV* self
    const char* func_name
    IV slot
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_op_rw_accessor(aTHX_ b, func_name, slot);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

void
inline_init()
    CODE:
        xs_jit_inline_init(aTHX);

int
inline_register(cv, type, slot, ...)
    CV* cv
    int type
    IV slot
    PREINIT:
        const char* key = NULL;
        STRLEN key_len = 0;
    CODE:
        /* Optional key for hash-based accessors */
        if (items > 3) {
            key = SvPV(ST(3), key_len);
        }
        RETVAL = xs_jit_inline_register(aTHX_ cv, (XS_JIT_InlineType)type, slot, key, key_len);
    OUTPUT:
        RETVAL

int
inline_get_type(cv)
    CV* cv
    CODE:
        RETVAL = (int)xs_jit_inline_get_type(aTHX_ cv);
    OUTPUT:
        RETVAL

# ============================================
# Custom op builder methods
# ============================================

SV*
pp_start(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_start(aTHX_ b, name);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_end(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_end(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_dsp(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_dsp(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_get_self(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_get_self(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_pop_self(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_pop_self(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_pop_sv(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_pop_sv(aTHX_ b, name);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_pop_nv(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_pop_nv(aTHX_ b, name);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_pop_iv(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_pop_iv(aTHX_ b, name);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_get_slots(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_get_slots(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_slot(self, name, slot)
    SV* self
    const char* name
    IV slot
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_slot(aTHX_ b, name, slot);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_return_sv(self, sv_expr)
    SV* self
    const char* sv_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_return_sv(aTHX_ b, sv_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_return_nv(self, nv_expr)
    SV* self
    const char* nv_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_return_nv(aTHX_ b, nv_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_return_iv(self, iv_expr)
    SV* self
    const char* iv_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_return_iv(aTHX_ b, iv_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_return_pv(self, pv_expr)
    SV* self
    const char* pv_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_return_pv(aTHX_ b, pv_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
pp_return(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_pp_return(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

# ============================================
# Call checker builder methods
# ============================================

SV*
ck_start(self, name)
    SV* self
    const char* name
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ck_start(aTHX_ b, name);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
ck_end(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ck_end(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
ck_preamble(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ck_preamble(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
ck_build_unop(self, pp_func, targ_expr)
    SV* self
    const char* pp_func
    const char* targ_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ck_build_unop(aTHX_ b, pp_func, targ_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
ck_build_binop(self, pp_func, targ_expr)
    SV* self
    const char* pp_func
    const char* targ_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ck_build_binop(aTHX_ b, pp_func, targ_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
ck_fallback(self)
    SV* self
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_ck_fallback(aTHX_ b);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

# ============================================
# XOP helpers
# ============================================

SV*
xop_declare(self, name, pp_func, desc)
    SV* self
    const char* name
    const char* pp_func
    const char* desc
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_xop_declare(aTHX_ b, name, pp_func, desc);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL

SV*
register_checker(self, cv_expr, ck_func, ckobj_expr)
    SV* self
    const char* cv_expr
    const char* ck_func
    const char* ckobj_expr
    PREINIT:
        XS_JIT_Builder* b;
    CODE:
        b = get_builder(aTHX_ self);
        xs_jit_register_checker(aTHX_ b, cv_expr, ck_func, ckobj_expr);
        RETVAL = SvREFCNT_inc(self);
    OUTPUT:
        RETVAL